/*=============================================================================
  (C)2023 HIL Lab.ALL RIGHT RESERVED.

  Redistribution of this file, in original or modified form, without
  prior written consent of HIL Lab is prohibited.

-------------------------------------------------------------------------------

  File:        attenuator_handler.h

  Description: Main include file for AttenuatorHandler API.

-------------------------------------------------------------------------------

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR IMPLIED
  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF TITLE,
  NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A PARTICULAR  PURPOSE ARE
  DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
  INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
  TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=============================================================================*/

#ifndef LIBATTENUATORHANDLER_INCLUDE_ATTENUATORHANDLER_H_
#define LIBATTENUATORHANDLER_INCLUDE_ATTENUATORHANDLER_H_

#ifdef LIBATTENUATORHANDLER_INCLUDE_ATTENUATORHANDLER_H_
#define LIBATTENUATORHANDLER __declspec(dllexport)
#else
#define LIBATTENUATORHANDLER __declspec(dllimport)
#endif

#include <string>
#include <memory>
#include <stdexcept>
#include <mutex>

namespace HILLAB {

  /**
  * Motor information
  **/
  struct MotorInfo {
    std::string serial_number;          // Motor serial number
    std::string year_of_manufacturing;  // Motor year of manufacturing
    std::string travel_degree;          // Motor travel degree
    std::string pulse_per_degree;       // Motor pulse per degree
    std::string firmware_release;       // Motor firmware release information
    std::string hardware_release;       // Motor hardware release information
    std::string mechanical_offset;      // Motor factory offset
  };

  enum class AttenuatorStatus {
    PARAMETER_OUT_OF_RANGE = -4,        // User input error
    SERIAL_CLOSED = -3,                 // Unable to write bytes to serial port
    SERIAL_TIMEOUT = -2,                // Unable to receive bytes from serial port
    SERIAL_UNDEFINED_ERROR = -1,        // Unexpected serial error
    OK = 0,                             // No problem
    MOTOR_COMMUNICATION_TIMEOUT = 1,    // Device internal error
    MOTOR_MECHANICAL_TIMEOUT = 2,       // Device internal error
    MOTOR_COMMAND_ERROR = 3,            // Device internal error
    MOTOR_VALUE_OUT_OF_RANGE = 4,       // Device internal error
    MOTOR_MODULE_ISOLATED = 5,          // Device internal error
    MOTOR_MODULE_OUT_OF_ISOLATION = 6,  // Device internal error
    MOTOR_INITIALIZING_ERROR = 7,       // Device internal error
    MOTOR_THERMAL_ERROR = 8,            // Device internal error
    MOTOR_BUSY = 9,                     // Device internal error
    MOTOR_SENSOR_ERROR = 10,            // Device internal error
    MOTOR_MOTOR_ERROR = 11,             // Device internal error
    MOTOR_OUT_OF_RANGE = 12,            // Device internal error
    MOTOR_OVER_CURRENT_ERROR = 13       // Device internal error
  };  

  class LIBATTENUATORHANDLER AttenuatorHandler {
  public:

    /**
    * Default constructor.
    * 
    **/
    AttenuatorHandler();

    /**
    * Default destructor. It calls Disconnect() when this invoked.
    * 
    **/
    ~AttenuatorHandler();

    /**
    * Connects device via RS232 interface. USB adapter acts just as electrical adapter. 
    * If the device is already connected, this call returns false and keeps the old device connected.
    * 
    * @param  device_port_name  Serial port. (e.g. "COM1")
    * @return Result of connection.
    * 
    * <Possible Error Code>
    *   AttenuatorStatus::OK (Regardless of whether the connection was successful or not)
    **/
    bool Connect(const std::string& device_port_name);

    /**
    * Disconnects device.
    * 
    * <Possible Error Code>
    *   AttenuatorStatus::OK
    **/
    void Disconnect();

    /**
    * Queries device status.
    * 
    * @param  serial_timeout  Serial communication timeout in milliseconds(defualt=3500).
    * @return Device status
    * <Possible Error Code>
    *   AttenuatorStatus::OK
    *   AttenuatorStatus::SERIAL_XX
    *   AttenuatorStatus::MOTOR_XX
    **/
    AttenuatorStatus GetCurrentStatus(const int serial_timeout = 3500);

    /**
    * Queries motor information.
    *
    * @param  out_info  Return value.
    * @param  serial_timeout  Serial communication timeout in milliseconds(defualt=7000).
    * @return Device status
    * <Possible Error Code>
    *   AttenuatorStatus::OK
    *   AttenuatorStatus::SERIAL_XX
    *   AttenuatorStatus::MOTOR_XX
    **/
    AttenuatorStatus GetMotorInfo(MotorInfo* out_info, const int serial_timeout = 7000);
    
    /**
     * Start mechanical calibration. It should be called once after power is applied to the product.
     * @param serial_timeout  Serial communication timeout in milliseconds(defualt=3500).
     * @return Device status
    * <Possible Error Code>
    *   AttenuatorStatus::OK
    *   AttenuatorStatus::SERIAL_XX
    *   AttenuatorStatus::MOTOR_XX
    */
    AttenuatorStatus MechanicalCalibration(const int serial_timeout = 3500);

    /**
    * Sets target transmittance value to run motor at the corresponding angle.
    * The offset degree cannot be less than -360, and the sum of the offset degree 
    * and range degree cannot be more than 360.
    * 
    * @param  transmittance Target transmittance value, range must be [0.0, 1.0]. 
    * @param  offset_degree Motor angle in degree when the transmittance is 0.
    * @param  range_degree  Angle in degree converted to transmittance range of 0-100%(default=45.0).
    *         That is, when the angle of the motor is @offset_degree + @range_degree, the transmittance will be maximum.
    * @param  serial_timeout  Serial communication timeout in milliseconds(defualt=3500).
    * @return Device status
    * <Possible Error Code>
    *   AttenuatorStatus::OK
    *   AttenuatorStatus::PARAMETER_OUT_OF_RANGE
    *   AttenuatorStatus::SERIAL_XX
    *   AttenuatorStatus::MOTOR_XX
    **/
    AttenuatorStatus SetTransmittance(const float transmittance, 
      const double offset_degree, const double range_degree = 45.0, 
      const int serial_timeout = 3500);

    /**
    * Queries current transmittance value.
    * The offset degree cannot be less than -360, and the sum of the offset degree 
    * and range degree cannot be more than 360.
    * 
    * @param  out_transmittance Return value, range might be [0.0, 1.0].
    * @param  offset_degree Motor angle in degree when the transmittance is 0.
    * @param  range_degree  Angle in degree converted to transmittance range of 0-100%(default=45.0).
    *         That is, when the angle of the motor is @offset_degree + @range_degree, the transmittance will be maximum.
    * @param  serial_timeout  Serial communication timeout in milliseconds(defualt=3500).
    * @return Device status
    * <Possible Error Code>
    *   AttenuatorStatus::OK
    *   AttenuatorStatus::SERIAL_XX
    *   AttenuatorStatus::MOTOR_XX
    **/
    AttenuatorStatus GetTransmittance(float* out_transmittance,
      const double offset_degree, const double range_degree = 45.0, 
      const int serial_timeout = 3500);

    /**
    * Sets target steps to run motor at the corresponding angle.
    * 
    * @param  steps Target stpes, range must be [-143360, 143360].
    * @param  serial_timeout  Serial communication timeout in milliseconds(defualt=3500).
    * @return Device status
    * <Possible Error Code>
    *   AttenuatorStatus::OK
    *   AttenuatorStatus::PARAMETER_OUT_OF_RANGE
    *   AttenuatorStatus::SERIAL_XX
    *   AttenuatorStatus::MOTOR_XX
    **/
    AttenuatorStatus SetMotorSteps(const int steps, const int serial_timeout = 3500);

    /**
    * Queries current steps.
    * 
    * @param  out_steps Return value, range might be [-143360, 143360].
    * @param  serial_timeout  Serial communication timeout in milliseconds(defualt=3500).
    * @return Device status
    * <Possible Error Code>
    *   AttenuatorStatus::OK
    *   AttenuatorStatus::SERIAL_XX
    *   AttenuatorStatus::MOTOR_XX
    **/
    AttenuatorStatus GetMotorSteps(int* out_steps, const int serial_timeout = 3500);

    /**
    * Sets target degree to run motor at the corresponding angle.
    * 
    * @param  degree  Target degree, range must be (-360.0, 360.0).
    * @param  serial_timeout  Serial communication timeout in milliseconds(defualt=3500).
    * @return Device status
    * <Possible Error Code>
    *   AttenuatorStatus::OK
    *   AttenuatorStatus::PARAMETER_OUT_OF_RANGE
    *   AttenuatorStatus::SERIAL_XX
    *   AttenuatorStatus::MOTOR_XX
    **/
    AttenuatorStatus SetMotorDegree(const float degree, const int serial_timeout = 3500);

    /**
    * Queries current rotary angle of the motor.
    * 
    * @param  out_degree  Return value, range might be (-360.0, 360.0)
    * @param  serial_timeout  Serial communication timeout in milliseconds(defualt=3500).
    * @return Device status
    * <Possible Error Code>
    *   AttenuatorStatus::OK
    *   AttenuatorStatus::SERIAL_XX
    *   AttenuatorStatus::MOTOR_XX
    **/
    AttenuatorStatus GetMotorDegree(float* out_degree, const int serial_timeout = 3500);

    /**
    * Optimizes the operating frequencies for backward and forward movement 
    * by searching frequencies and monitoring mechanical performances. 
    * Allowing 20 minutes cool down period is highly recommended after optimization.
    * This operation may take several minutes, 
    * during that time the device and the associated communication bus are blocked. 
    * In this case, AttenuatorStatus::MOTOR_BUSY is thrown when another method is called, 
    * except Stop(). 
    * 
    * @param  serial_timeout  Serial communication timeout in milliseconds(defualt=3500).
    * @return Device status
    * <Possible Error Code>
    *   AttenuatorStatus::SERIAL_XX
    *   AttenuatorStatus::MOTOR_XX
    **/
    AttenuatorStatus StartOptimization(const int serial_timeout = 3500);

    /**
    * Starts cleaning cycle, during which the device will move backwards and forwards 
    * over its full range of travel for a few munutes, in order to remove any dust or debris 
    * from the bearing rails and motor contacts.
    * Allowing 20 minutes cool down period is highly recommended after optimization.
    * This operation may take several minutes, 
    * during that time the device and the associated communication bus are blocked. 
    * In this case, AttenuatorStatus::MOTOR_BUSY is thrown when another method is called, 
    * except Stop(). 
    * 
    * @param  serial_timeout  Serial communication timeout in milliseconds(defualt=3500).
    * @return Device status
    * <Possible Error Code>
    *   AttenuatorStatus::SERIAL_XX
    *   AttenuatorStatus::MOTOR_XX
    **/
    AttenuatorStatus StartCleaningMechanics(const int serial_timeout = 3500);

    /**
    * Aborts the optimization or cleaning process initiated by 
    * StartOptimization() or StartCleaningMechanics().
    * 
    * @param  serial_timeout  Serial communication timeout in milliseconds(defualt=3500).
    * @return Device status
    * <Possible Error Code>
    *   AttenuatorStatus::OK
    *   AttenuatorStatus::SERIAL_XX
    *   AttenuatorStatus::MOTOR_XX
    **/
    AttenuatorStatus Stop(const int serial_timeout = 3500);

  private:
    class __Member__;
    std::unique_ptr<__Member__> m;
  };

};

#endif  // LIBATTENUATORHANDLER_INCLUDE_ATTENUATORHANDLER_H_